// Mobile Navigation Toggle
const navToggle = document.getElementById('nav-toggle');
const navMenu = document.getElementById('nav-menu');
const navLinks = document.querySelectorAll('.nav-link');

navToggle.addEventListener('click', () => {
    navMenu.classList.toggle('active');

    // Animate Icon
    const icon = navToggle.querySelector('i');
    if (navMenu.classList.contains('active')) {
        icon.classList.remove('fa-bars');
        icon.classList.add('fa-xmark');
    } else {
        icon.classList.remove('fa-xmark');
        icon.classList.add('fa-bars');
    }
});

// Close Mobile Menu when a link is clicked
navLinks.forEach(link => {
    link.addEventListener('click', () => {
        navMenu.classList.remove('active');
        const icon = navToggle.querySelector('i');
        icon.classList.remove('fa-xmark');
        icon.classList.add('fa-bars');
    });
});

// Navbar Background Change on Scroll
const navbar = document.querySelector('.navbar');

window.addEventListener('scroll', () => {
    if (window.scrollY > 50) {
        navbar.style.background = 'rgba(15, 23, 42, 0.95)';
        navbar.style.boxShadow = '0 10px 30px rgba(0,0,0,0.1)';
    } else {
        navbar.style.background = 'rgba(15, 23, 42, 0.7)';
        navbar.style.boxShadow = 'none';
    }
});

// Active Link Highlight on Scroll
const sections = document.querySelectorAll('section');

window.addEventListener('scroll', () => {
    let current = '';

    sections.forEach(section => {
        const sectionTop = section.offsetTop;
        const sectionHeight = section.clientHeight;

        // 150px offset for the fixed header
        if (pageYOffset >= (sectionTop - 150)) {
            current = section.getAttribute('id');
        }
    });

    navLinks.forEach(link => {
        link.classList.remove('active');
        if (link.getAttribute('href').includes(current)) {
            link.classList.add('active');
        }
    });
});

// --- Dynamic Canvas Background (Network/Constellation Effect) ---
const canvas = document.getElementById('network-bg');
if (canvas) {
    const ctx = canvas.getContext('2d');
    let width, height;
    let particles = [];

    // Configuration (Responsive)
    const isMobile = window.innerWidth < 768;
    const particleCount = isMobile ? 40 : 100; // Much fewer on mobile
    const connectionDistance = isMobile ? 120 : 180; // Shorter connections on mobile
    const mouseDistance = 200;

    // Resize canvas
    function resize() {
        width = canvas.width = canvas.parentElement.offsetWidth;
        height = canvas.height = canvas.parentElement.offsetHeight;
    }

    window.addEventListener('resize', resize);
    resize();

    // Particle Class
    class Particle {
        constructor() {
            this.x = Math.random() * width;
            this.y = Math.random() * height;
            this.vx = (Math.random() - 0.5) * 1.5; // Velocity X
            this.vy = (Math.random() - 0.5) * 1.5; // Velocity Y
            this.size = isMobile ? 1.5 : (Math.random() * 2 + 1);
            this.color = isMobile ? `rgba(94, 255, 0, 0.15)` : `rgba(94, 255, 0, ${Math.random() * 0.2 + 0.2})`;
        }

        update() {
            this.x += this.vx;
            this.y += this.vy;

            // Bounce off edges
            if (this.x < 0 || this.x > width) this.vx *= -1;
            if (this.y < 0 || this.y > height) this.vy *= -1;
        }

        draw() {
            ctx.beginPath();
            ctx.arc(this.x, this.y, this.size, 0, Math.PI * 2);
            ctx.fillStyle = this.color;
            ctx.fill();
        }
    }

    // Initialize particles
    for (let i = 0; i < particleCount; i++) {
        particles.push(new Particle());
    }

    // Animation Loop
    function animate() {
        ctx.clearRect(0, 0, width, height);

        // Update and draw particles
        particles.forEach((p, index) => {
            p.update();
            p.draw();

            // Connect particles
            for (let j = index + 1; j < particles.length; j++) {
                const p2 = particles[j];
                const dx = p.x - p2.x;
                const dy = p.y - p2.y;
                const distance = Math.sqrt(dx * dx + dy * dy);

                if (distance < connectionDistance) {
                    ctx.beginPath();
                    ctx.strokeStyle = `rgba(94, 255, 0, ${(1 - distance / connectionDistance) * 0.8})`; // Brighter Electric Lime connections
                    ctx.lineWidth = 0.8;
                    ctx.moveTo(p.x, p.y);
                    ctx.lineTo(p2.x, p2.y);
                    ctx.stroke();
                }
            }
        });

        requestAnimationFrame(animate);
    }

    animate();
}

// --- Number Counter Animation ---
const stats = document.querySelectorAll('.stat-item h3');
let ranStats = false; // Run only once

const statsObserver = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting && !ranStats) {
            ranStats = true;
            stats.forEach(stat => {
                const finalValueText = stat.innerText;
                const finalValue = parseFloat(finalValueText.replace(/[^0-9.]/g, ''));
                const suffix = finalValueText.replace(/[0-9.]/g, '');
                const duration = 2000; // 2 seconds
                const frameDuration = 1000 / 60; // 60fps
                const totalFrames = Math.round(duration / frameDuration);
                let frame = 0;

                const counter = setInterval(() => {
                    frame++;
                    const progress = frame / totalFrames;
                    // Ease out function
                    const currentVal = finalValue * (1 - Math.pow(1 - progress, 3));

                    // Format output
                    if (finalValue % 1 !== 0) {
                        stat.innerText = currentVal.toFixed(1) + suffix;
                    } else {
                        stat.innerText = Math.round(currentVal) + suffix;
                    }

                    if (frame === totalFrames) {
                        clearInterval(counter);
                        stat.innerText = finalValueText; // Ensure exact final value
                    }
                }, frameDuration);
            });
        }
    });
}, { threshold: 0.5 });

const statsSection = document.querySelector('.hero-stats');
if (statsSection) statsObserver.observe(statsSection);


// --- Simple Scroll Reveal (Vanilla JS) ---
document.addEventListener("DOMContentLoaded", () => {
    // Select elements to animate
    const revealElements = document.querySelectorAll('.service-card, .pricing-card, .section-header, .hero-content > *, .hero-image, .contact-form, .info-item, .hero-stats .stat-item, .area-grid > *');

    const revealOptions = {
        threshold: 0.1, // Trigger when 10% of element is visible
        rootMargin: "0px 0px -50px 0px"
    };

    const revealOnScroll = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('visible');
                // Optional: Stop observing once revealed
                observer.unobserve(entry.target);
            }
        });
    }, revealOptions);

    revealElements.forEach(el => {
        el.classList.add('reveal-hidden');
        revealOnScroll.observe(el);
    });
});

// Inject styles for reveal animation dynamically
const style = document.createElement('style');
style.innerHTML = `
    .reveal-hidden {
        opacity: 0;
        transform: translateY(30px);
        transition: all 0.8s cubic-bezier(0.5, 0, 0, 1);
    }
    .visible {
        opacity: 1 !important;
        transform: translateY(0) !important;
    }
`;
document.head.appendChild(style);

// --- Live Speed Meter Animation ---
// --- Live Speed Meter Animation (English) ---
const liveSpeedEl = document.getElementById('live-speed');

if (liveSpeedEl) {
    let currentSpeed = 0;
    const targetSpeed = 100;

    function animateSpeed() {
        if (currentSpeed < targetSpeed) {
            currentSpeed += Math.floor(Math.random() * 5) + 1;
            if (currentSpeed > targetSpeed) currentSpeed = targetSpeed;

            liveSpeedEl.innerText = currentSpeed;
            setTimeout(animateSpeed, 20);
        } else {
            fluctuateSpeed();
        }
    }

    function fluctuateSpeed() {
        const fluctuation = Math.floor(Math.random() * (100 - 95 + 1) + 95);
        liveSpeedEl.innerText = fluctuation;
        setTimeout(fluctuateSpeed, Math.random() * (800 - 200) + 200);
    }

    setTimeout(animateSpeed, 500);
}
// --- Mobile Menu Logic (Enhanced Sidebar) ---
const toggle = document.getElementById('nav-toggle');
const backdrop = document.getElementById('mobile-menu-overlay');
const closeBtn = document.getElementById('close-menu');

if (toggle && backdrop && closeBtn) {
    toggle.addEventListener('click', () => {
        backdrop.classList.add('active');
    });

    closeBtn.addEventListener('click', () => {
        backdrop.classList.remove('active');
    });

    // Close when clicking outside sidebar
    backdrop.addEventListener('click', (e) => {
        if (e.target === backdrop) {
            backdrop.classList.remove('active');
        }
    });
}

// --- Preloader Logic ---
window.addEventListener('load', () => {
    const preloader = document.getElementById('preloader');
    if (preloader) {
        // Minimum display time for effect
        setTimeout(() => {
            preloader.style.opacity = '0';
            preloader.style.visibility = 'hidden';
        }, 800);
    }
});
// --- Router Visual Speed (Why Upgrade Section) ---
const routerSpeedVal = document.getElementById('router-speed-val');

if (routerSpeedVal) {
    let rSpeed = 0;
    const rTarget = 100;

    function animateRouterSpeed() {
        if (rSpeed < rTarget) {
            rSpeed += Math.floor(Math.random() * 3) + 1;
            if (rSpeed > rTarget) rSpeed = rTarget;
            routerSpeedVal.innerText = rSpeed; // English digits
            setTimeout(animateRouterSpeed, 30);
        } else {
            fluctuateRouterSpeed();
        }
    }

    function fluctuateRouterSpeed() {
        // High Speed Fluctuation (Hyper Active)
        const flux = Math.floor(Math.random() * (103 - 97 + 1) + 97);
        routerSpeedVal.innerText = flux;
        setTimeout(fluctuateRouterSpeed, Math.random() * (200 - 50) + 50); // Very fast updates
    }

    // Start when in view (simplified: start on load for now)
    setTimeout(animateRouterSpeed, 800);
}
